/*
  Antenna Rotator Rurn Indicator
  Yaesu G-800A version

  Glen Popiel - KW5GP

  Uses Adafruit NeoPixel Library

  Released under the GPLv3 license
*/

//#define debug // Uncomment this to enable the display of debug information on the Serial Monitor

#include <Adafruit_NeoPixel.h>  // Include the Adafruit NeoPixel library

#define led_pin 6 // Define the Neopixel I/O pin

#define num_pixels 12 // NeoPixel Strip/Ring size

#define ccw_pin 3 //define the counterclockwise rotation input pin
#define cw_pin 4 // define the clockwise rotation input pin
#define debounce 20 // Define the switch debounce delay

// Setup the NeoPixel library
// Uncomment the following for standard RGB Neopixel/WS281x LED Strips
//Adafruit_NeoPixel pixels(num_pixels, led_pin, NEO_GRB + NEO_KHZ800);

// Uncomment the following for RGBW Neopixel/WS281x LED Strips (RGB w/White LED)
// If you look at the unlit LEDs, half of each LED is covered with yellow
Adafruit_NeoPixel pixels(num_pixels, led_pin, NEO_GRBW + NEO_KHZ800);

#define delay_time 500 // Time (in milliseconds) to pause between pixels

int current_pixel, last_pixel; // pixel variables
bool cw_in, ccw_in, moving; // Rotation variables


void setup()
{
#ifdef debug
  Serial.begin(9600); // Start the serial port if debugging is enabled
#endif

  pinMode(cw_pin, INPUT_PULLUP);  // Set the pin modes for the controller inputs
  pinMode(ccw_pin, INPUT_PULLUP);

  pixels.begin(); // Initialize the NeoPixel display
  pixels.clear(); // Set all pixel colors to 'off'
  pixels.show();   // Send the updated pixel colors to the hardware
  moving = false; // Clear the moving flag
}

void loop()
{
  cw_in = digitalRead(cw_pin);  // Read the CW and CCW inputs
  ccw_in = digitalRead(ccw_pin);
  delay(debounce);

  if ((cw_in == LOW || ccw_in == LOW))  // If one of them is low, we're rotating
  {
#ifdef debug
    Serial.print("   CCW = ");
    Serial.print(ccw_in);
    Serial.print("   CW = ");
    Serial.println(cw_in);
    Serial.println("CW or CCW Switch pressed");
#endif

    moving = true;  // Set the rotation flag
    
    if (cw_in == LOW) // If the cw input is low, we're turning clockwise (right)
    {
      // we're turning clockwise

#ifdef debug
      Serial.println("Turning Clockwise");
#endif

      turn_cw();  // Call the turn_cw() function
    }

    if (ccw_in == LOW) // If the ccw input is low, we're turning counterclockwise (left)
    {
      // We're turning counterclockwise

#ifdef debug
      Serial.println("Turning CounterClockwise");
#endif

      turn_ccw();  // Call the turn_ccw() function
    }
  } else
  {
    if (moving) // Have we been moving and aren't anymore? If so, flash the brake lights
    {
      moving = false; // Turn off the rotation flag
    }
    //nothing happening- clear the LED's
    pixels.clear(); // Set all pixel colors to 'off'
    pixels.show();   // Send the updated pixel colors to the hardware
  }
}
                                                                                                                                                                                                                                                                                                                                                                                                                      
void turn_cw()  // The turn_cw function
{
  int current_pixel = 0;  // Set the starting LED point
  delay(debounce);
  while (digitalRead(cw_pin) == LOW)  // Repeat the loop as long as the cw switch is pressed
  {
    pixels.setPixelColor(current_pixel, pixels.Color(0, 150, 0));    // For each pixel set LED color to green
    pixels.show();   // Send the updated pixel colors to the hardware.

    if (current_pixel >= num_pixels)  // If we've lit all of the LED's in the ring
    {
      // clear all pixels
      pixels.clear(); // Set all pixel colors to 'off'
      pixels.show();   // Send the updated pixel colors to the hardware
      current_pixel = current_pixel - num_pixels; // Start over with the first pixel
    } else
    {
      current_pixel = current_pixel + 1;  // Otherwise, add an LED
    }
    delay(delay_time); // Pause before next pass through loop
  }
  brake();  // Call the brake() function with the switch is released
}


void turn_ccw()  // The turn_ccw function
{
  int current_pixel = num_pixels - 1; // Set the starting LED point
  delay(debounce);
  while (digitalRead(ccw_pin) == LOW)  // Repeat the loop as long as the cw switch is pressed
  {
    pixels.setPixelColor(current_pixel, pixels.Color(0, 150, 0)); // For each pixel set LED color to green
    pixels.show();   // Send the updated pixel colors to the hardware.

    if (current_pixel < 0)  // If we've lit all of the LEDs
    {
      // clear all pixels
      pixels.clear(); // Set all pixel colors to 'off'
      pixels.show();   // Send the updated pixel colors to the hardware

      current_pixel = num_pixels - 1; // Reset the starting point
    } else
    {
      current_pixel = current_pixel - 1;  // Select the next LED to light
    }
    delay(delay_time); // Pause before next pass through loop
  }
  brake();  // Call the brake() function with the switch is released
}

void brake()  // The brake() function
{
#ifdef debug
  Serial.println("Braking");
#endif

  pixels.fill(0, 0, num_pixels);  // Turn off all the LED's
  pixels.show();   // Send the updated pixel colors to the hardware.
  delay(delay_time);
  for (int i = 0; i < 5; i = i + 1)  // Flash all LEDs for 5 seconds for braking indicator
  {
    pixels.fill(pixels.Color(150, 0, 0), 0, num_pixels);  // Set the LED color to red
    pixels.show();   // Send the updated pixel colors to the hardware.
    delay(delay_time * 2);
    pixels.fill(0, 0, num_pixels);
    pixels.show();   // Send the updated pixel colors to the hardware.
    delay(delay_time * 2);
  }
}

